import { Router } from "express";
import {
  createUpdateBcpRequirementAnalysis,
  deleteBcpRequirementAnalysis,
  getBcpRequirementAnalysis,
} from "../controller/bcpRequirementAnalysisController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

const BcpRequirementAnalysisRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: BCP Requirement Analysis
 *   description: API endpoints for managing BCP Requirement Analysis
 */

/**
 * @swagger
 * /api/bcpRequirementAnalysis/create-update-bcp-requirement-analysis:
 *   post:
 *     summary: Create or update a BCP Requirement Analysis
 *     tags:
 *       - BCP Requirement Analysis
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               location:
 *                 type: integer
 *               tasks:
 *                 type: string
 *               assessment_date:
 *                 type: string
 *                 format: date
 *               lead_Assessor:
 *                 type: string
 *               assessment_participants:
 *                 type: string
 *               overall_impact_score:
 *                 type: integer
 *               aggregate_activity_score:
 *                 type: integer
 *               does_this_activity_occur_in_other_locations:
 *                 type: string
 *               recovery_time_value:
 *                 type: integer
 *               recovery_time_duration:
 *                 type: string
 *               maximum_tolerable_period_of_disruption_value:
 *                 type: integer
 *               maximum_tolerable_period_of_disruption_duration:
 *                 type: string
 *               rating_explanation:
 *                 type: integer
 *           example:
 *             name: Business Process A
 *             description: Analysis of key business process
 *             location: 1
 *             tasks: "Task 1, Task 2"
 *             assessment_date: "2024-11-20"
 *             lead_Assessor: John Doe
 *             assessment_participants: "Jane Smith, Mike Johnson"
 *             overall_impact_score: 85
 *             aggregate_activity_score: 90
 *             does_this_activity_occur_in_other_locations: "Yes, Location B"
 *             recovery_time_value: 48
 *             recovery_time_duration: "Hours"
 *             maximum_tolerable_period_of_disruption_value: 72
 *             maximum_tolerable_period_of_disruption_duration: "Hours"
 *             rating_explanation: 3
 *     responses:
 *       '200':
 *         description: BCP Requirement Analysis created or updated successfully
 *       '401':
 *         description: Unauthorized, invalid token
 *       '500':
 *         description: Server error
 */
BcpRequirementAnalysisRoutes.post(
  "/create-update-bcp-requirement-analysis",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      433,
      "bcp_requirement_analysis"
    )(req, res, next);
  },
  controllerWrapper(createUpdateBcpRequirementAnalysis)
);

/**
 * @swagger
 * /api/bcpRequirementAnalysis/get-bcp-requirement-analysis:
 *   get:
 *     summary: Get all BCP Requirement Analyses
 *     tags:
 *       - BCP Requirement Analysis
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       '200':
 *         description: Successfully retrieved BCP Requirement Analyses
 *       '401':
 *         description: Unauthorized, invalid token
 *       '500':
 *         description: Server error
 */
BcpRequirementAnalysisRoutes.get(
  "/get-bcp-requirement-analysis",
  verifyToken,
  sideBarPermissionCheck("view", 433, "bcp_requirement_analysis"),
  controllerWrapper(getBcpRequirementAnalysis)
);

/**
 * @swagger
 * /api/bcpRequirementAnalysis/get-bcp-requirement-analysis/{id}:
 *   get:
 *     summary: Get a BCP Requirement Analysis by ID
 *     tags:
 *       - BCP Requirement Analysis
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: ID of the BCP Requirement Analysis to retrieve
 *     responses:
 *       '200':
 *         description: Successfully retrieved the BCP Requirement Analysis
 *       '401':
 *         description: Unauthorized, invalid token
 *       '404':
 *         description: BCP Requirement Analysis not found
 *       '500':
 *         description: Server error
 */
BcpRequirementAnalysisRoutes.get(
  "/get-bcp-requirement-analysis/:id",
  verifyToken,
  sideBarPermissionCheck("view", 433, "bcp_requirement_analysis"),
  controllerWrapper(getBcpRequirementAnalysis)
);

/**
 * @swagger
 * /api/bcpRequirementAnalysis/delete-bcp-requirement-analysis/{id}:
 *   delete:
 *     summary: Delete a BCP Requirement Analysis
 *     tags:
 *       - BCP Requirement Analysis
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: ID of the BCP Requirement Analysis to delete
 *     responses:
 *       '200':
 *         description: BCP Requirement Analysis deleted successfully
 *       '401':
 *         description: Unauthorized, invalid token
 *       '404':
 *         description: BCP Requirement Analysis not found
 *       '500':
 *         description: Server error
 */
BcpRequirementAnalysisRoutes.delete(
  "/delete-bcp-requirement-analysis/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 433, "bcp_requirement_analysis"),
  controllerWrapper(deleteBcpRequirementAnalysis)
);

export default BcpRequirementAnalysisRoutes;




