import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { createUpdateCategory, deleteCategory, getParentCategoryList, viewCategory, viewCategoryDropDown } from "../controller/categoryController.js";

/**
 * @swagger
 * tags:
 *   name:  category
 *   description: API operations for managing  category
 */
const categoryRoutes = Router();

/**
 * @swagger
 * /api/category/create-update-category:
 *   post:
 *     summary: Create or update a  category
 *     tags: [ category]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example: { id: "id", title: "title", description: "description" }
 */
categoryRoutes.post(
  "/create-update-category",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 405 , "category")(req, res, next);
  },
  controllerWrapper(createUpdateCategory)
);

/**
 * @swagger
 * /api/category/get-category:
 *   get:
 *     summary: Get all records of  category
 *     tags: [ category]
 */
categoryRoutes.get(
  "/get-category",
  verifyToken,
  sideBarPermissionCheck("view", 405, "category"),
  controllerWrapper(viewCategory)
);

/**
 * @swagger
 * /api/category/get-category/{id}:
 *   get:
 *     summary: Get a specific record of  category
 *     tags: [ category]
 */
categoryRoutes.get(
  "/get-category/:id",
  verifyToken,
  // sideBarPermissionCheck("view", 405, "category"),
  controllerWrapper(viewCategory)
);

/**
 * @swagger
 * /api/category/get-disability-dropdown:
 *   get:
 *     summary: Get the list all Category
 *     tags:
 *       - Category
 *     description: The Category
 *     responses:
 *       200:
 *         description: Category data
 *       404:
 *         description: Category not found
 *       500:
 *         description: Internal server error
 */
categoryRoutes.get(
  "/get-category-dropdown",
  verifyToken,
  // sideBarPermissionCheck("view", 405, "category"),
  controllerWrapper(viewCategoryDropDown)
);

/**
 * @swagger
 * /api/category/get-category-list:
 *   get:
 *     summary: Get the list for selecting parent category
 *     tags: [Disability]
 *     parameters:
 *       - in: query
 *         organization : [1,2,34,4]
 *         schema:
 *           type: Array
 *         description: Array of organization to fetch the list
 */

categoryRoutes.get("/get-category-list", verifyToken, controllerWrapper(getParentCategoryList));

/**
 * @swagger
 * /api/category/delete-category/{id}:
 *   delete:
 *     summary: Delete a specific record of  category
 *     tags: [ category]
 */
categoryRoutes.delete(
  "/delete-category/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 405, "category"),
  controllerWrapper(deleteCategory)
);

export default categoryRoutes;