import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdateContractorSiteAccess,
  getContractorSiteAccess,
  deleteContractorSiteAccess,
  uploadDocument,
} from "../controller/contractorSiteAccessController.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Contractor SiteAccess
 *   description: API operations for managing contractor site access
 */

/**
 * @swagger
 * components:
 *   schemas:
 *     ContractorSiteAccess:
 *       type: object
 *       properties:
 *         id:
 *           type: integer
 *           description: The auto-generated ID of the contractor site access
 *         contractor:
 *           type: integer
 *           description: ID of the contractor
 *         status:
 *           type: string
 *           description: The current status of the site access
 *         form:
 *           type: string
 *           description: The form data in long text format
 *         organization:
 *           type: integer
 *           description: ID of the associated organization
 *         department:
 *           type: integer
 *           description: ID of the associated department
 *         deleted:
 *           type: integer
 *           description: Whether the site access is deleted (0 or 1)
 *         created_by:
 *           type: integer
 *           description: ID of the user who created the site access
 *         created_at:
 *           type: string
 *           format: date-time
 *           description: The date and time when the site access was created
 *         updated_by:
 *           type: integer
 *           description: ID of the user who last updated the site access
 *         updated_at:
 *           type: string
 *           format: date-time
 *           description: The date and time when the site access was last updated
 */

const contractorSiteAccessRoutes = Router();

/**
 * @swagger
 * /api/contractorSiteAccess/create-update-contractor-site-access:
 *   post:
 *     summary: Create or update a contractor site access
 *     tags: [Contractor SiteAccess]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ContractorSiteAccess'
 *     responses:
 *       200:
 *         description: The contractor site access was successfully created/updated
 */

contractorSiteAccessRoutes.post(
  "/create-update-contractor-site-access",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      269,
      "contractor_site_access"
    )(req, res, next);
  },
  controllerWrapper(createUpdateContractorSiteAccess)
);

/**
 * @swagger
 * /api/contractorSiteAccess/get-contractor-site-access:
 *   get:
 *     summary: Get all contractor site access
 *     tags: [Contractor SiteAccess]
 *     responses:
 *       200:
 *         description: List of all contractor site access
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/ContractorSiteAccess'
 */

contractorSiteAccessRoutes.get(
  "/get-contractor-site-access",
  verifyToken,
  sideBarPermissionCheck("view", 269, "contractor_site_access"),
  controllerWrapper(getContractorSiteAccess)
);

/**
 * @swagger
 * /api/contractorSiteAccess/get-contractor-site-access/{id}:
 *   get:
 *     summary: Get a contractor site access by ID
 *     tags: [Contractor SiteAccess]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the contractor site access
 *     responses:
 *       200:
 *         description: Contractor site access data
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ContractorSiteAccess'
 */


contractorSiteAccessRoutes.get(
  "/get-contractor-site-access/:id",
  verifyToken,
  sideBarPermissionCheck("view", 269, "contractor_site_access"),
  controllerWrapper(getContractorSiteAccess)
);

/**
 * @swagger
 * /api/contractorSiteAccess/delete-contractor-site-access/{id}:
 *   delete:
 *     summary: Delete a contractor site access by ID
 *     tags: [Contractor SiteAccess]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the contractor site access to delete
 *     responses:
 *       200:
 *         description: Contractor site access was deleted
 */

contractorSiteAccessRoutes.delete(
  "/delete-contractor-site-access/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 269, "contractor_site_access"),
  controllerWrapper(deleteContractorSiteAccess)
);

/**
 * @swagger
 * /api/contractorSiteAccess/upload-site-access:
 *   post:
 *     summary: Create or update a contractor pre site
 *     tags: [Contractor SiteAccess]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             id : 1
 *             document : "file"
 *     responses:
 *       200:
 *         description: Document uploaded successfully
 */

contractorSiteAccessRoutes.post(
  "/upload-site-access",
  verifyToken,
  sideBarPermissionCheck("edit", 269, "contractor_site_access"),
  controllerWrapper(uploadDocument)
);

export default contractorSiteAccessRoutes;
