import { Router } from "express";
import { controllerWrapper } from "../helper/wrapper.js";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  bcpComment,
  createUpdateBcpManagement,
  deleteBcpManagement,
  getBcpManagement,
} from "../controller/bcpManagementController.js";
import {
  createUpdateCrisisManagement,
  deleteCrisisManagement,
  getCrisisManagement,
} from "../controller/crisisManagementController.js";

const CrisisManagementRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Crisis Management
 *   description: API endpoints for managing Crisis Management
 */

/**
 * @swagger
 * /api/crisisManagement/create-update-crisis-management:
 *   post:
 *     summary: Create or update a Crisis Management entry
 *     tags:
 *       - Crisis Management
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               responsible_person:
 *                 type: string
 *               tasks:
 *                 type: string
 *               timeline:
 *                 type: string
 *                 format: date
 *               status:
 *                 type: string
 *           example:
 *             name: Emergency Plan Development
 *             description: Development of the company's emergency response plan
 *             responsible_person: Jane Doe
 *             tasks: "Plan writing, Resource allocation"
 *             timeline: "2024-12-15"
 *             status: "In Progress"
 *     responses:
 *       '200':
 *         description: Crisis Management entry created or updated successfully
 *       '401':
 *         description: Unauthorized, invalid token
 *       '500':
 *         description: Server error
 */
CrisisManagementRoutes.post(
  "/create-update-crisis-management",
  verifyToken,
  // (req, res, next) => {
  //   sideBarPermissionCheck(
  //     req.body.id ? "edit" : "create",
  //     434,
  //     "bcp_management"
  //   )(req, res, next);
  // },
  controllerWrapper(createUpdateCrisisManagement)
);

/**
 * @swagger
 * /api/crisisManagement/get-crisis-management:
 *   get:
 *     summary: Get all Crisis Management entries
 *     tags:
 *       - Crisis Management
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       '200':
 *         description: Successfully retrieved Crisis Management entries
 *       '401':
 *         description: Unauthorized, invalid token
 *       '500':
 *         description: Server error
 */
CrisisManagementRoutes.get(
  "/get-crisis-management",
  verifyToken,
  // (req, res, next) => {
  //   sideBarPermissionCheck("view", 434, "crisis_management")(req, res, next);
  // },
  controllerWrapper(getCrisisManagement)
);

/**
 * @swagger
 * /api/crisisManagement/get-crisis-management/{id}:
 *   get:
 *     summary: Get a Crisis Management entry by ID
 *     tags:
 *       - Crisis Management
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: ID of the Crisis Management entry to retrieve
 *     responses:
 *       '200':
 *         description: Successfully retrieved the Crisis Management entry
 *       '401':
 *         description: Unauthorized, invalid token
 *       '404':
 *         description: Crisis Management entry not found
 *       '500':
 *         description: Server error
 */
CrisisManagementRoutes.get(
  "/get-crisis-management/:id",
  verifyToken,
  // (req, res, next) => {
  //   sideBarPermissionCheck("view", 434, "bcp_management")(req, res, next);
  // },
  controllerWrapper(getCrisisManagement)
);

/**
 * @swagger
 * /api/crisisManagement/delete-crisis-management/{id}:
 *   delete:
 *     summary: Delete a Crisis Management entry by ID
 *     tags:
 *       - Crisis Management
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: ID of the Crisis Management entry to delete
 *     responses:
 *       '200':
 *         description: Crisis Management entry deleted successfully
 *       '401':
 *         description: Unauthorized, invalid token
 *       '404':
 *         description: Crisis Management entry not found
 *       '500':
 *         description: Server error
 */
CrisisManagementRoutes.delete(
  "/delete-crisis-management/:id",
  verifyToken,
  // (req, res, next) => {
  //   sideBarPermissionCheck("delete", 434, "bcp_management")(req, res, next);
  // },
  controllerWrapper(deleteCrisisManagement)
);

export default CrisisManagementRoutes;
