import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import {
  createExplosionBombForm,
  getExplosionBombForm,
  deleteExplosionBombForm,
} from "../controller/explosionBombFormController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

/**
 * @swagger
 * tags:
 *   name: Explosion Bomb Form
 *   description: API operations for managing Explosion Bomb Form
 */
const explosionBombFormRoutes = Router();

/**
 * @swagger
 * /api/explosionBombForm/create-explosion-bomb-form:
 *   post:
 *     summary: Create a new Explosion Bomb Form
 *     tags: [Explosion Bomb Form]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               type_of_incident:
 *                 type: string
 *               location:
 *                 type: string
 *               incident_id:
 *                 type: integer
 *               category_id:
 *                 type: integer
 *               immediate_cause_of_incident:
 *                 type: string
 *               was_alcohol_or_substance_testing_done:
 *                 type: boolean
 *               tested_by:
 *                 type: string
 *               state_name:
 *                 type: string
 *               alcohol_substance_test_results:
 *                 type: string
 *               method_of_testing:
 *                 type: string
 *               was_the_person_or_operator_licensed:
 *                 type: boolean
 *               was_the_license_applicable_to_the_task:
 *                 type: boolean
 *               license_training_expiry_date:
 *                 type: string
 *                 format: date
 *               license_no_training_certificate_no:
 *                 type: string
 *               year_issued:
 *                 type: string
 *               issued_by:
 *                 type: string
 *               incident_level:
 *                 type: string
 *               reportable_cases_classification:
 *                 type: string
 *               was_the_case_reported_to_authority:
 *                 type: boolean
 *               comments:
 *                 type: string
 *               form_completed_by:
 *                 type: string
 *               designation:
 *                 type: string
 *               date:
 *                 type: string
 *                 format: date
 *               organization:
 *                 type: integer
 *               department:
 *                 type: integer
 *           example:
 *             type_of_incident: "Explosion"
 *             location: "Warehouse"
 *             incident_id: 123
 *             category_id: 1
 *             immediate_cause_of_incident: "Gas Leak"
 *             was_alcohol_or_substance_testing_done: true
 *             tested_by: "John Doe"
 *             state_name: "State"
 *             alcohol_substance_test_results: "Negative"
 *             method_of_testing: "Breathalyzer"
 *             was_the_person_or_operator_licensed: true
 *             was_the_license_applicable_to_the_task: true
 *             license_training_expiry_date: "2023-12-31"
 *             license_no_training_certificate_no: "LIC12345"
 *             year_issued: "2022"
 *             issued_by: "Authority"
 *             incident_level: "High"
 *             reportable_cases_classification: "Critical"
 *             was_the_case_reported_to_authority: true
 *             comments: "Initial report submitted"
 *             form_completed_by: "John Doe"
 *             designation: "Manager"
 *             date: "2023-01-01"
 *             organization: 1
 *             department: 2
 *     responses:
 *       200:
 *         description: Record created successfully
 *       404:
 *         description: Cannot create record
 */
explosionBombFormRoutes.post(
  "/create-explosion-bomb-form",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      176,
      "explosion_bomb_form"
    )(req, res, next);
  },
  controllerWrapper(createExplosionBombForm)
);

/**
 * @swagger
 * /api/explosionBombForm/get-explosion-bomb-form/{id}:
 *   get:
 *     summary: Get a specific Explosion Bomb Form by ID
 *     tags: [Explosion Bomb Form]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the Explosion Bomb Form
 *     responses:
 *       200:
 *         description: Explosion Bomb Form retrieved successfully
 *       404:
 *         description: Explosion Bomb Form not found
 */
explosionBombFormRoutes.get(
  "/get-explosion-bomb-form/:id",
  verifyToken,
  sideBarPermissionCheck("view", 176, "explosion_bomb_form"),
  controllerWrapper(getExplosionBombForm)
);

/**
 * @swagger
 * /api/explosionBombForm/delete-explosion-bomb-form/{id}:
 *   delete:
 *     summary: Delete a specific Explosion Bomb Form by ID
 *     tags: [Explosion Bomb Form]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the Explosion Bomb Form to delete
 *     responses:
 *       200:
 *         description: Record deleted successfully
 *       404:
 *         description: Cannot delete record
 */
explosionBombFormRoutes.delete(
  "/delete-explosion-bomb-form/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 176, "explosion_bomb_form"),
  controllerWrapper(deleteExplosionBombForm)
);

export default explosionBombFormRoutes;
