// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdatedIssuingAuthority,
  deleteIssuingAuthority,
  viewAllIssuingAuthority,
} from "../controller/issuingAuthorityController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { validateRequest } from "../helper/general.js";
import { issuingAuthorityCreateUpdateValidationSchema } from "../validationsRecord/validationSchema.js";

/**
 * @swagger
 * tags:
 *   name: Issuing Authority
 *   description: API operations for managing all Issuing Authorities
 */
const issuingAuthorityRoutes = Router();

/**
 * @swagger
 * /api/issuingAuthority/create-update-issuing-authority:
 *   post:
 *     summary: Create or update an Issuing Authority
 *     tags: [Issuing Authority]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             id: 1
 *             name: "Authority Name"
 *             description: "Description of the issuing authority"
 *             organization: [1, 2]
 *     responses:
 *       200:
 *         description: Issuing Authority created or updated successfully
 *       400:
 *         description: Bad request
 *       500:
 *         description: Internal server error
 */
issuingAuthorityRoutes.post(
  "/create-update-issuing-authority",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      149,
      "issuing_authority"
    )(req, res, next);
  },
  validateRequest(issuingAuthorityCreateUpdateValidationSchema),
  controllerWrapper(createUpdatedIssuingAuthority)
);

/**
 * @swagger
 * /api/issuingAuthority/get-issuing-authority:
 *   get:
 *     summary: Get all list of Issuing Authorities
 *     tags: [Issuing Authority]
 *     responses:
 *       200:
 *         description: A list of issuing authorities
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Internal server error
 */
issuingAuthorityRoutes.get(
  "/get-issuing-authority",
  verifyToken,
  sideBarPermissionCheck("view", 149, "issuing_authority"),
  controllerWrapper(viewAllIssuingAuthority)
);

/**
 * @swagger
 * /api/issuingAuthority/get-issuing-authority/{id}:
 *   get:
 *     summary: Get a specific record of Issuing Authority
 *     tags: [Issuing Authority]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the issuing authority
 *     responses:
 *       200:
 *         description: A specific issuing authority record
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Not found
 *       500:
 *         description: Internal server error
 */
issuingAuthorityRoutes.get(
  "/get-issuing-authority/:id",
  verifyToken,
  sideBarPermissionCheck("view", 149, "issuing_authority"),
  controllerWrapper(viewAllIssuingAuthority)
);

/**
 * @swagger
 * /api/issuingAuthority/delete-issuing-authority/{id}:
 *   delete:
 *     summary: Delete a specific record of Issuing Authority
 *     tags: [Issuing Authority]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the issuing authority to delete
 *     responses:
 *       200:
 *         description: Issuing Authority deleted successfully
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Not found
 *       500:
 *         description: Internal server error
 */
issuingAuthorityRoutes.delete(
  "/delete-issuing-authority/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 149, "issuing_authority"),
  controllerWrapper(deleteIssuingAuthority)
);

export default issuingAuthorityRoutes;
