// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import {
  createUpdateNameOfStandard,
  deleteNameOfStandard,
  viewNameOfStandard,
} from "../controller/nameOfStandardController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Name Of standard
 *   description: API operations for managing all Name Of standard
 */
const nameOfStandardRoutes = Router();

/**
 * @swagger
 * /api/nameOfStandard/create-update-name-of-standard:
 *   post:
 *     summary: Create or update an Name Of standard
 *     tags: [Name Of standard]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *           example:
 *             id: "id"
 *             name: "Name of standard "
 *             description: "Description of the name of standard"
 *             organization: ["Organization1", "Organization2"]
 *     responses:
 *       200:
 *         description: Name of standard created or updated successfully
 *       400:
 *         description: Bad request. Invalid input parameters
 *       500:
 *         description: Internal server error
 */
nameOfStandardRoutes.post(
  "/create-update-name-of-standard",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      312,
      "name_of_standard"
    )(req, res, next);
  },
  controllerWrapper(createUpdateNameOfStandard)
);

/**
 * @swagger
 * /api/nameOfStandard/get-name-of-standard:
 *   get:
 *     summary: Get list of all Name Of standard
 *     tags: [Name Of standard]
 *     responses:
 *       200:
 *         description: List of name of standard
 *       500:
 *         description: Internal server error
 */
nameOfStandardRoutes.get(
  "/get-name-of-standard",
  verifyToken,
  sideBarPermissionCheck("view", 312, "name_of_standard"),
  controllerWrapper(viewNameOfStandard)
);

/**
 * @swagger
 * /api/nameOfStandard/get-name-of-standard/{id}:
 *   get:
 *     summary: Get a specific record of Name Of standard
 *     tags: [Name Of standard]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the name of standard
 *     responses:
 *       200:
 *         description: Name of standard data
 *       404:
 *         description: Name of standard not found
 *       500:
 *         description: Internal server error
 */
nameOfStandardRoutes.get(
  "/get-name-of-standard/:id",
  verifyToken,
  sideBarPermissionCheck("view", 312, "name_of_standard"),
  controllerWrapper(viewNameOfStandard)
);

/**
 * @swagger
 * /api/nameOfStandard/delete-name-of-standard/{id}:
 *   delete:
 *     summary: Delete a specific record of Name Of standard
 *     tags: [Name Of standard]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the insurance policy impact to delete
 *     responses:
 *       200:
 *         description: Name of standard successfully deleted
 *       404:
 *         description: Name of standard not found
 *       500:
 *         description: Internal server error
 */
nameOfStandardRoutes.delete(
  "/delete-name-of-standard/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 312, "name_of_standard"),
  controllerWrapper(deleteNameOfStandard)
);

export default nameOfStandardRoutes;
