import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

import {
  createOccupationalDiseaseForm,
  getOccupationalDiseaseForm,
  deleteOccupationalDiseaseForm,
} from "../controller/occupationalDiseaseTechnicalFormController.js";

const occupationalDiseaseFormRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Occupational Diseases
 *   description: API operations for managing occupational disease forms
 */

/**
 * @swagger
 * /api/occupationalDiseaseForm/create-occupational-disease-form:
 *   post:
 *     summary: Create a new occupational disease form
 *     tags: [Occupational Diseases]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               category_id:
 *                 type: integer
 *               incident_id:
 *                 type: integer
 *               name_of_exposed_person:
 *                 type: string
 *               role:
 *                 type: integer
 *               were_they_licensed_trained:
 *                 type: string
 *               was_license_training_applicable_to_task:
 *                 type: string
 *               license_training_expiry_date:
 *                 type: string
 *                 format: date
 *               license_no_training_certificate_no:
 *                 type: string
 *               year:
 *                 type: string
 *               issued_by:
 *                 type: string
 *               occupational_disease_category:
 *                 type: string
 *               noise_induced_hearing_loss:
 *                 type: string
 *               date_of_diagnosis:
 *                 type: string
 *                 format: date
 *               description:
 *                 type: string
 *               was_case_reported_to_authority_regulator:
 *                 type: string
 *               name_of_regulator:
 *                 type: string
 *               date_reported:
 *                 type: string
 *                 format: date
 *               reported_by_who:
 *                 type: string
 *               effect_on_person:
 *                 type: string
 *               exposure_route:
 *                 type: string
 *               source_of_exposure:
 *                 type: string
 *               estimated_duration_of_exposure:
 *                 type: string
 *               other_employees_exposed:
 *                 type: string
 *               names_of_exposed_employees:
 *                 type: string
 *               occupational_hygiene_survey_conducted:
 *                 type: string
 *               occupational_hygiene_survey_date:
 *                 type: string
 *                 format: date
 *               estimated_cost_medical_expenses:
 *                 type: number
 *                 format: float
 *               comments:
 *                 type: string
 *               form_completed_by:
 *                 type: string
 *               designation:
 *                 type: string
 *               signature:
 *                 type: string
 *               date:
 *                 type: string
 *                 format: date
 *           example:
 *             category_id: 1
 *             incident_id: 123
 *             name_of_exposed_person: "John Doe"
 *             role: 2
 *             were_they_licensed_trained: "Yes"
 *             was_license_training_applicable_to_task: "Yes"
 *             license_training_expiry_date: "2024-01-01"
 *             license_no_training_certificate_no: "ABC123"
 *             year: "2023"
 *             issued_by: "Health Authority"
 *             occupational_disease_category: "Respiratory"
 *             noise_induced_hearing_loss: "No"
 *             date_of_diagnosis: "2023-05-10"
 *             description: "Details of the case."
 *             was_case_reported_to_authority_regulator: "Yes"
 *             name_of_regulator: "Local Health Authority"
 *             date_reported: "2023-06-15"
 *             reported_by_who: "Dr. Smith"
 *             effect_on_person: "Temporary"
 *             exposure_route: "Inhalation"
 *             source_of_exposure: "Chemical"
 *             estimated_duration_of_exposure: "6 months"
 *             other_employees_exposed: "3"
 *             names_of_exposed_employees: "Jane Doe, Mark Smith"
 *             occupational_hygiene_survey_conducted: "Yes"
 *             occupational_hygiene_survey_date: "2023-07-01"
 *             estimated_cost_medical_expenses: 1500.00
 *             comments: "Follow-up required."
 *             form_completed_by: "Alice"
 *             designation: "Health Officer"
 *             signature: "Alice Signature"
 *             date: "2023-07-10"
 *     responses:
 *       200:
 *         description: The created occupational disease form
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 id:
 *                   type: integer
 *                 category_id:
 *                   type: integer
 *                 incident_id:
 *                   type: integer
 *                 name_of_exposed_person:
 *                   type: string
 *                 role:
 *                   type: integer
 *                 were_they_licensed_trained:
 *                   type: string
 *                 was_license_training_applicable_to_task:
 *                   type: string
 *                 license_training_expiry_date:
 *                   type: string
 *                   format: date
 *                 license_no_training_certificate_no:
 *                   type: string
 *                 year:
 *                   type: string
 *                 issued_by:
 *                   type: string
 *                 occupational_disease_category:
 *                   type: string
 *                 noise_induced_hearing_loss:
 *                   type: string
 *                 date_of_diagnosis:
 *                   type: string
 *                   format: date
 *                 description:
 *                   type: string
 *                 was_case_reported_to_authority_regulator:
 *                   type: string
 *                 name_of_regulator:
 *                   type: string
 *                 date_reported:
 *                   type: string
 *                   format: date
 *                 reported_by_who:
 *                   type: string
 *                 effect_on_person:
 *                   type: string
 *                 exposure_route:
 *                   type: string
 *                 source_of_exposure:
 *                   type: string
 *                 estimated_duration_of_exposure:
 *                   type: string
 *                 other_employees_exposed:
 *                   type: string
 *                 names_of_exposed_employees:
 *                   type: string
 *                 occupational_hygiene_survey_conducted:
 *                   type: string
 *                 occupational_hygiene_survey_date:
 *                   type: string
 *                   format: date
 *                 estimated_cost_medical_expenses:
 *                   type: number
 *                   format: float
 *                 comments:
 *                   type: string
 *                 form_completed_by:
 *                   type: string
 *                 designation:
 *                   type: string
 *                 signature:
 *                   type: string
 *                 date:
 *                   type: string
 *                   format: date
 *       400:
 *         description: Bad request
 *       500:
 *         description: Internal server error
 */
occupationalDiseaseFormRoutes.post(
  "/create-occupational-disease-form",
  verifyToken,
  controllerWrapper(createOccupationalDiseaseForm)
);

/**
 * @swagger
 * /api/occupationalDiseaseForm/get-occupational-disease-form/{id}:
 *   get:
 *     summary: Get an occupational disease form by ID
 *     tags: [Occupational Diseases]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The occupational disease form ID
 *     responses:
 *       200:
 *         description: The occupational disease form data
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 id:
 *                   type: integer
 *                 category_id:
 *                   type: integer
 *                 incident_id:
 *                   type: integer
 *                 name_of_exposed_person:
 *                   type: string
 *                 role:
 *                   type: integer
 *                 were_they_licensed_trained:
 *                   type: string
 *                 was_license_training_applicable_to_task:
 *                   type: string
 *                 license_training_expiry_date:
 *                   type: string
 *                   format: date
 *                 license_no_training_certificate_no:
 *                   type: string
 *                 year:
 *                   type: string
 *                 issued_by:
 *                   type: string
 *                 occupational_disease_category:
 *                   type: string
 *                 noise_induced_hearing_loss:
 *                   type: string
 *                 date_of_diagnosis:
 *                   type: string
 *                   format: date
 *                 description:
 *                   type: string
 *                 was_case_reported_to_authority_regulator:
 *                   type: string
 *                 name_of_regulator:
 *                   type: string
 *                 date_reported:
 *                   type: string
 *                   format: date
 *                 reported_by_who:
 *                   type: string
 *                 effect_on_person:
 *                   type: string
 *                 exposure_route:
 *                   type: string
 *                 source_of_exposure:
 *                   type: string
 *                 estimated_duration_of_exposure:
 *                   type: string
 *                 other_employees_exposed:
 *                   type: string
 *                 names_of_exposed_employees:
 *                   type: string
 *                 occupational_hygiene_survey_conducted:
 *                   type: string
 *                 occupational_hygiene_survey_date:
 *                   type: string
 *                   format: date
 *                 estimated_cost_medical_expenses:
 *                   type: number
 *                   format: float
 *                 comments:
 *                   type: string
 *                 form_completed_by:
 *                   type: string
 *                 designation:
 *                   type: string
 *                 signature:
 *                   type: string
 *                 date:
 *                   type: string
 *                   format: date
 *       400:
 *         description: Bad request
 *       404:
 *         description: Occupational disease form not found
 *       500:
 *         description: Internal server error
 */
occupationalDiseaseFormRoutes.get(
  "/get-occupational-disease-form/:id",
  verifyToken,
  controllerWrapper(getOccupationalDiseaseForm)
);

/**
 * @swagger
 * /api/occupationalDiseaseForm/delete-occupational-disease-form/{id}:
 *   delete:
 *     summary: Delete an occupational disease form by ID
 *     tags: [Occupational Diseases]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The occupational disease form ID
 *     responses:
 *       200:
 *         description: The occupational disease form was deleted
 *       400:
 *         description: Bad request
 *       404:
 *         description: Occupational disease form not found
 *       500:
 *         description: Internal server error
 */
occupationalDiseaseFormRoutes.delete(
  "/delete-occupational-disease-form/:id",
  verifyToken,
  controllerWrapper(deleteOccupationalDiseaseForm)
);

export default occupationalDiseaseFormRoutes;
