import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdateOhsFileRequirement,
  getOhsFileRequirement,
  deleteOhsFileRequirement,
} from "../controller/ohsFileRequirementController.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Contractor PreSite
 *   description: API operations for managing ohs file requirement
 */
/**
 * @swagger
 * components:
 *   schemas:
 *     OhsFileRequirement:
 *       type: object
 *       properties:
 *         id:
 *           type: integer
 *           description: The auto-generated ID of the OHS file requirement.
 *         contractor:
 *           type: integer
 *           description: The contractor ID.
 *         company_registration_no:
 *           type: string
 *           description: Company registration number.
 *         coida_registration_no:
 *           type: string
 *           description: COIDA registration number.
 *         contact_person:
 *           type: string
 *           description: Contact person for the OHS file.
 *         contact_number:
 *           type: string
 *           description: Contact number of the person responsible.
 *         site_address:
 *           type: string
 *           description: The physical site address for the contractor.
 *         postal_address:
 *           type: string
 *           description: The postal address for the contractor.
 *         section_appointment:
 *           type: string
 *           description: Section appointment details.
 *         health_safety_representative:
 *           type: string
 *           description: Information on the health and safety representative.
 *         activities_rendered:
 *           type: string
 *           description: Activities rendered by the contractor.
 *         commencement_date:
 *           type: string
 *           format: date
 *           description: The start date of the project or contract.
 *         completion_date:
 *           type: string
 *           format: date
 *           description: The projected completion date of the project.
 *         employees_on_site:
 *           type: string
 *           description: The total number of employees on site.
 *         female:
 *           type: integer
 *           description: The number of female employees on site.
 *         male:
 *           type: integer
 *           description: The number of male employees on site.
 *         people_with_disabilities:
 *           type: string
 *           description: Number of people with disabilities.
 *         organization:
 *           type: integer
 *           description: The associated organization ID.
 *         department:
 *           type: integer
 *           description: The associated department ID.
 *         deleted:
 *           type: integer
 *           description: Whether the OHS file is marked as deleted (0 or 1).
 *         created_by:
 *           type: integer
 *           description: ID of the user who created the OHS file.
 *         created_at:
 *           type: string
 *           format: date-time
 *           description: The timestamp when the OHS file was created.
 *         updated_by:
 *           type: integer
 *           description: ID of the user who last updated the OHS file.
 *         updated_at:
 *           type: string
 *           format: date-time
 *           description: The timestamp when the OHS file was last updated.
 */

const ohsFileRequirementRoutes = Router();
/**
 * @swagger
 * /api/ohsFileRequirement/create-update-ohs-file-requirements:
 *   post:
 *     summary: Create or update an OHS file requirement.
 *     tags: [Ohs File Requirement]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/OhsFileRequirement'
 *     responses:
 *       200:
 *         description: The OHS file requirement was successfully created/updated.
 */

ohsFileRequirementRoutes.post(
  "/create-update-ohs-file-requirements",
//   verifyToken,
  controllerWrapper(createUpdateOhsFileRequirement)
);

/**
 * @swagger
 * /api/ohsFileRequirement/get-ohs-file-requirements:
 *   get:
 *     summary: Retrieve all OHS file requirements.
 *     tags: [Ohs File Requirement]
 *     responses:
 *       200:
 *         description: A list of OHS file requirements.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/OhsFileRequirement'
 */

ohsFileRequirementRoutes.get(
  "/get-ohs-file-requirements",
  verifyToken,
  controllerWrapper(getOhsFileRequirement)
);

/**
 * @swagger
 * /api/ohsFileRequirement/get-ohs-file-requirements/{id}:
 *   get:
 *     summary: Get an OHS file requirement by ID.
 *     tags: [Ohs File Requirement]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the OHS file requirement.
 *     responses:
 *       200:
 *         description: An OHS file requirement.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/OhsFileRequirement'
 */

ohsFileRequirementRoutes.get(
  "/get-ohs-file-requirements/:id",
  verifyToken,
  controllerWrapper(getOhsFileRequirement)
);

/**
 * @swagger
 * /api/ohsFileRequirement/delete-ohs-file-requirements/{id}:
 *   delete:
 *     summary: Delete an OHS file requirement by ID.
 *     tags: [Ohs File Requirement]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the OHS file requirement to delete.
 *     responses:
 *       200:
 *         description: The OHS file requirement was deleted successfully.
 */

ohsFileRequirementRoutes.delete(
  "/delete-ohs-file-requirements/:id",
  verifyToken,
  controllerWrapper(deleteOhsFileRequirement)
);

export default ohsFileRequirementRoutes;
