// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  deleteOperationalRiskRegister,
  operationalRiskRegisterCreateUpdate,
  viewOperationalRiskRegister,
} from "../controller/operationalRiskRegisterController.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Operational Risk Register
 *   description: API Operations For Managing Operational Risks
 */
const operationalRiskRegisterRoutes = Router();

/**
 * @swagger
 * /api/operationalRisk/create-update-operational-risk:
 *   post:
 *     summary: Create Or Update An Operational Risk
 *     tags: [Operational Risk Register]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: integer
 *               risk_register_name:
 *                 type: string
 *               assessment_date:
 *                 type: string
 *                 format: date
 *               review_date:
 *                 type: string
 *                 format: date
 *               risk_context:
 *                 type: string
 *               risk_assessment_facilitator:
 *                 type: string
 *               risk_register_approver:
 *                 type: string
 *               meeting_id:
 *                 type: integer
 *               organization:
 *                 type: string
 *               department:
 *                 type: string
 *           example:
 *             id: 1
 *             risk_register_name: "Operational Risk Register 2024"
 *             assessment_date: "2024-01-15"
 *             review_date: "2024-06-15"
 *             risk_context: "Context for the operational risk."
 *             risk_assessment_facilitator: "John Doe"
 *             risk_register_approver: "Jane Smith"
 *             meeting_id: 1001
 *             organization: "Company A"
 *             department: "Risk Management"
 */
operationalRiskRegisterRoutes.post(
  "/create-update-operational-risk",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      38,
      "operational_risk"
    )(req, res, next);
  },
  controllerWrapper(operationalRiskRegisterCreateUpdate)
);

/**
 * @swagger
 * /api/operationalRisk/get-operational-risk:
 *   get:
 *     summary: Get list of all operational risks
 *     tags: [Operational Risk Register]
 */
operationalRiskRegisterRoutes.get(
  "/get-operational-risk",
  verifyToken,
  sideBarPermissionCheck("view", 38, "operational_risk"),
  controllerWrapper(viewOperationalRiskRegister)
);

/**
 * @swagger
 * /api/operationalRisk/get-operational-risk/{id}:
 *   get:
 *     summary: Get A Specific Operational Risk Record
 *     tags: [Operational Risk Register]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The operational risk ID
 */
operationalRiskRegisterRoutes.get(
  "/get-operational-risk/:id",
  verifyToken,
  sideBarPermissionCheck("view", 38, "operational_risk"),
  controllerWrapper(viewOperationalRiskRegister)
);

/**
 * @swagger
 * /api/operationalRisk/delete-operational-risk/{id}:
 *   delete:
 *     summary: Delete a specific record of operational risk
 *     tags: [Operational Risk Register]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The operational risk ID
 */
operationalRiskRegisterRoutes.delete(
  "/delete-operational-risk/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 38, "operational_risk"),
  controllerWrapper(deleteOperationalRiskRegister)
);

export default operationalRiskRegisterRoutes;
