// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import {
  createUpdateRoleOnTheProject,
  deleteRoleOnTheProject,
  viewRoleOnTheProject,
} from "../controller/roleOnTheProjectController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Role On the project
 *   description: API operations for managing all Role On the project
 */
const roleOnTheProjectRoutes = Router();

/**
 * @swagger
 * /api/roleOnTheProject/create-update-role-on-project:
 *   post:
 *     summary: Create or update an Role On the project
 *     tags: [Role On the project]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *           example:
 *             id: "id"
 *             name: "Policy Impact Name"
 *             description: "Description of the role on the project"
 *             organization: ["Organization1", "Organization2"]
 *     responses:
 *       200:
 *         description: Role On the project created or updated successfully
 *       400:
 *         description: Bad request. Invalid input parameters
 *       500:
 *         description: Internal server error
 */
roleOnTheProjectRoutes.post(
  "/create-update-role-on-project",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      326,
      "role_on_the_project"
    )(req, res, next);
  },
  controllerWrapper(createUpdateRoleOnTheProject)
);

/**
 * @swagger
 * /api/roleOnTheProject/get-role-on-project:
 *   get:
 *     summary: Get list of all RoleOnTheProject
 *     tags: [Role On the project]
 *     responses:
 *       200:
 *         description: List of roleOnTheProject
 *       500:
 *         description: Internal server error
 */
roleOnTheProjectRoutes.get(
  "/get-role-on-project",
  verifyToken,
  sideBarPermissionCheck("view", 326, "role_on_the_project"),
  controllerWrapper(viewRoleOnTheProject)
);

/**
 * @swagger
 * /api/roleOnTheProject/get-role-on-project/{id}:
 *   get:
 *     summary: Get a specific record of Role On the project
 *     tags: [Role On the project]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the role on the project
 *     responses:
 *       200:
 *         description: Role On the project data
 *       404:
 *         description: Role On the project not found
 *       500:
 *         description: Internal server error
 */
roleOnTheProjectRoutes.get(
  "/get-role-on-project/:id",
  verifyToken,
  sideBarPermissionCheck("view", 326, "role_on_the_project"),
  controllerWrapper(viewRoleOnTheProject)
);

/**
 * @swagger
 * /api/roleOnTheProject/delete-role-on-project/{id}:
 *   delete:
 *     summary: Delete a specific record of Role On the project
 *     tags: [Role On the project]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the role on the project to delete
 *     responses:
 *       200:
 *         description: Role On the project successfully deleted
 *       404:
 *         description: Role On the project not found
 *       500:
 *         description: Internal server error
 */
roleOnTheProjectRoutes.delete(
  "/delete-role-on-project/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 326, "role_on_the_project"),
  controllerWrapper(deleteRoleOnTheProject)
);

export default roleOnTheProjectRoutes;
