// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import {
  createUpdateSidebar,
  deleteSidebar,
  viewSidebar,
} from "../controller/sidebarController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Sidebar
 *   description: API operations for managing all Sidebar
 */
const sidebarRoutes = Router();

/**
 * @swagger
 * /api/sidebar/create-update-sidebar:
 *   post:
 *     summary: Create or update an Sidebar
 *     tags: [Sidebar]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *           example:
 *             id: "id"
 *             name: "Policy Impact Name"
 *             description: "Description of the sidebar"
 *             organization: ["Organization1", "Organization2"]
 *     responses:
 *       200:
 *         description: Sidebar created or updated successfully
 *       400:
 *         description: Bad request. Invalid input parameters
 *       500:
 *         description: Internal server error
 */
sidebarRoutes.post(
  "/create-update-sidebar",
  verifyToken,
  controllerWrapper(createUpdateSidebar)
);

/**
 * @swagger
 * /api/sidebar/get-sidebar:
 *   get:
 *     summary: Get list of all Sidebar
 *     tags: [Sidebar]
 *     responses:
 *       200:
 *         description: List of sidebar
 *       500:
 *         description: Internal server error
 */
sidebarRoutes.get(
  "/get-sidebar",
  verifyToken,
  controllerWrapper(viewSidebar)
);

/**
 * @swagger
 * /api/sidebar/get-sidebar/{id}:
 *   get:
 *     summary: Get a specific record of Sidebar
 *     tags: [Sidebar]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the sidebar
 *     responses:
 *       200:
 *         description: Sidebar data
 *       404:
 *         description: Sidebar not found
 *       500:
 *         description: Internal server error
 */
sidebarRoutes.get(
  "/get-sidebar/:id",
  verifyToken,
  controllerWrapper(viewSidebar)
);

/**
 * @swagger
 * /api/sidebar/delete-sidebar/{id}:
 *   delete:
 *     summary: Delete a specific record of Sidebar
 *     tags: [Sidebar]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the sidebar to delete
 *     responses:
 *       200:
 *         description: Sidebar successfully deleted
 *       404:
 *         description: Sidebar not found
 *       500:
 *         description: Internal server error
 */
sidebarRoutes.delete(
  "/delete-sidebar/:id",
  verifyToken,
  controllerWrapper(deleteSidebar)
);

export default sidebarRoutes;
