import { Router } from "express";
import {
  createUpdateActivityRecording,
  getStakeholderActivityRecording,
  deleteStakeholderActivityRecording,
  approveRejectStakeholderActivityRecording,
  fetchRecordingScheduleData,
  feedbackAnalysis,
  // fetchRecordingScheduleData,
} from "../controller/stakeholderActivityRecordingController.js";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";

//sidebar id : 243
/**
 * @swagger
 * tags:
 *   name: Stakeholder Activity Recording
 *   description: API operations for managing stakeholder activity recordings
 */

const stakeholderActivityRecordingRoutes = Router();

/**
 * @swagger
 * /api/stakeholderActivityRecording/create-update-activity-recording:
 *   post:
 *     summary: Create or update a stakeholder activity recording
 *     tags: [Stakeholder Activity Recording]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: integer
 *                 example: 1
 *               activity_chairperson:
 *                 type: integer
 *                 example: 5
 *               activity_scriber:
 *                 type: integer
 *                 example: 8
 *               activity_duration:
 *                 type: string
 *                 example: "2 hours"
 *               actual_activity_date_from:
 *                 type: string
 *                 format: date-time
 *                 example: "2024-07-30T09:00:00Z"
 *               actual_activity_date_to:
 *                 type: string
 *                 format: date-time
 *                 example: "2024-07-30T11:00:00Z"
 *               participants:
 *                 type: string
 *                 example: "John Doe, Jane Smith"
 *               scheduled_activity:
 *                 type: integer
 *                 example: 2
 *               status:
 *                 type: string
 *                 example: "Completed"
 *               activity_notes:
 *                 type: JSON
 *                 example: [{} , {} , {}]
 *               organization:
 *                 type: integer
 *                 example: 3
 *               department:
 *                 type: integer
 *                 example: 4
 *               deleted:
 *                 type: integer
 *                 example: 0
 *               created_by:
 *                 type: integer
 *                 example: 1
 *               created_at:
 *                 type: string
 *                 format: date-time
 *               updated_by:
 *                 type: integer
 *                 example: 2
 *               updated_at:
 *                 type: string
 *                 format: date-time
 *     responses:
 *       200:
 *         description: Stakeholder activity recording created or updated successfully
 *       400:
 *         description: Bad request
 */
stakeholderActivityRecordingRoutes.post(
  "/create-update-activity-recording",
  verifyToken,
  controllerWrapper(createUpdateActivityRecording)
);

/**
 * @swagger
 * /api/stakeholderActivityRecording/get-activity-recording:
 *   get:
 *     summary: Get all stakeholder activity recordings
 *     tags: [Stakeholder Activity Recording]
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       200:
 *         description: A list of stakeholder activity recordings
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 type: object
 *                 properties:
 *                   id:
 *                     type: integer
 *                     example: 1
 *                   activity_chairperson:
 *                     type: integer
 *                     example: 5
 *                   activity_scriber:
 *                     type: integer
 *                     example: 8
 *                   activity_duration:
 *                     type: string
 *                     example: "2 hours"
 *                   actual_activity_date_from:
 *                     type: string
 *                     format: date-time
 *                     example: "2024-07-30T09:00:00Z"
 *                   actual_activity_date_to:
 *                     type: string
 *                     format: date-time
 *                     example: "2024-07-30T11:00:00Z"
 *                   participants:
 *                     type: string
 *                     example: "John Doe, Jane Smith"
 *                   scheduled_activity:
 *                     type: integer
 *                     example: 2
 *                   status:
 *                     type: string
 *                     example: "Completed"
 *                   organization:
 *                     type: integer
 *                     example: 3
 *                   department:
 *                     type: integer
 *                     example: 4
 *                   deleted:
 *                     type: integer
 *                     example: 0
 *                   created_by:
 *                     type: integer
 *                     example: 1
 *                   created_at:
 *                     type: string
 *                     format: date-time
 *                   updated_by:
 *                     type: integer
 *                     example: 2
 *                   updated_at:
 *                     type: string
 *                     format: date-time
 *       400:
 *         description: Bad request
 */
stakeholderActivityRecordingRoutes.get(
  "/get-activity-recording",
  verifyToken,
  controllerWrapper(getStakeholderActivityRecording)
);

/**
 * @swagger
 * /api/stakeholderActivityRecording/get-activity-recording/{id}:
 *   get:
 *     summary: Get a stakeholder activity recording by ID
 *     tags: [Stakeholder Activity Recording]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the stakeholder activity recording
 *     responses:
 *       200:
 *         description: A stakeholder activity recording
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 id:
 *                   type: integer
 *                   example: 1
 *                 activity_chairperson:
 *                   type: integer
 *                   example: 5
 *                 activity_scriber:
 *                   type: integer
 *                   example: 8
 *                 activity_duration:
 *                   type: string
 *                   example: "2 hours"
 *                 actual_activity_date_from:
 *                   type: string
 *                   format: date-time
 *                   example: "2024-07-30T09:00:00Z"
 *                 actual_activity_date_to:
 *                   type: string
 *                   format: date-time
 *                   example: "2024-07-30T11:00:00Z"
 *                 participants:
 *                   type: string
 *                   example: "John Doe, Jane Smith"
 *                 scheduled_activity:
 *                   type: integer
 *                   example: 2
 *                 status:
 *                   type: string
 *                   example: "Completed"
 *                 organization:
 *                   type: integer
 *                   example: 3
 *                 department:
 *                   type: integer
 *                   example: 4
 *                 deleted:
 *                   type: integer
 *                   example: 0
 *                 created_by:
 *                   type: integer
 *                   example: 1
 *                 created_at:
 *                   type: string
 *                   format: date-time
 *                 updated_by:
 *                   type: integer
 *                   example: 2
 *                 updated_at:
 *                   type: string
 *                   format: date-time
 *       400:
 *         description: Bad request
 *       404:
 *         description: Stakeholder activity recording not found
 */
stakeholderActivityRecordingRoutes.get(
  "/get-activity-recording/:id",
  verifyToken,
  controllerWrapper(getStakeholderActivityRecording)
);

/**
 * @swagger
 * /api/stakeholderActivityRecording/delete-activity-recording/{id}:
 *   delete:
 *     summary: Delete a stakeholder activity recording by ID
 *     tags: [Stakeholder Activity Recording]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the stakeholder activity recording to delete
 *     responses:
 *       200:
 *         description: Stakeholder activity recording deleted successfully
 *       400:
 *         description: Bad request
 *       404:
 *         description: Stakeholder activity recording not found
 */
stakeholderActivityRecordingRoutes.delete(
  "/delete-activity-recording/:id",
  verifyToken,
  controllerWrapper(deleteStakeholderActivityRecording)
);

/**
 * @swagger
 * /api/stakeholderActivityRecording/approve-reject-recording:
 *   post:
 *     summary: Approve or reject stakeholder activity recording
 *     tags: [Stakeholder Activity Recording]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: integer
 *                 example: 1
 *               approval_status:
 *                 type: string
 *                 example: "Approved/Rejected"
 *     responses:
 *       200:
 *         description: Stakeholder activity recording approved or rejected successfully
 *       400:
 *         description: Bad request
 */

stakeholderActivityRecordingRoutes.post(
  "/approve-reject-recording",
  verifyToken,
  controllerWrapper(approveRejectStakeholderActivityRecording)
);

/**
 * @swagger
 * /api/stakeholderActivityRecording/get-recording-calendar-data:
 *   get:
 *     summary: Download the file for the calendar
 *     tags: [Stakeholder Activity Recording]
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       200:
 *         description: File path of the ics file
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 type: object
 *       400:
 *         description: Bad request
 */

stakeholderActivityRecordingRoutes.get(
  "/get-recording-calendar-data",
  // verifyToken,
  controllerWrapper(fetchRecordingScheduleData)
);

stakeholderActivityRecordingRoutes.get(
  "/get-feedback-analysis-data",
  verifyToken,
  controllerWrapper(feedbackAnalysis)
);

export default stakeholderActivityRecordingRoutes;
