import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import {
  createUpdateStakeholderIdentification,
  getStakeholderIdentification,
  deleteStakeholderIdentification,
  getInternalStakeHolders,
} from "../controller/stakeholderIdentificationController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

const stakeholderIdentificationRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Stakeholder Identification
 *   description: API operations for managing stakeholder identification
 */

/**
 * @swagger
 * components:
 *   schemas:
 *     StakeHolderIdentification:
 *       type: object
 *       properties:
 *         id:
 *           type: integer
 *           description: The auto-generated id of the stakeholder identification
 *         stakeholder_type:
 *           type: integer
 *           description: Type of stakeholder
 *         stakeholder_category:
 *           type: string
 *           description: Category of stakeholder
 *         stakeholder_department_internal:
 *           type: integer
 *           description: Internal department of stakeholder
 *         stakeholder_name_internal:
 *           type: integer
 *           description: Internal name of stakeholder
 *         stakeholder_name_external:
 *           type: string
 *           description: External name of stakeholder
 *         role_external:
 *           type: string
 *           description: External role of stakeholder
 *         email:
 *           type: string
 *           description: Email of stakeholder
 *         phone:
 *           type: string
 *           description: Phone number of stakeholder
 *         address:
 *           type: string
 *           description: Address of stakeholder
 *         classification:
 *           type: string
 *           description: Classification of stakeholder
 *         engagement_level:
 *           type: string
 *           description: Engagement level of stakeholder
 *         status:
 *           type: string
 *           description: Status of stakeholder
 *         relationship_type:
 *           type: string
 *           description: Relationship type of stakeholder
 *         interaction_frequency:
 *           type: string
 *           description: Interaction frequency of stakeholder
 *         influence_level:
 *           type: string
 *           description: Influence level of stakeholder
 *         connection_strength:
 *           type: string
 *           description: Connection strength of stakeholder
 *         interest_score:
 *           type: string
 *           description: Interest score of stakeholder
 *         influence_score:
 *           type: string
 *           description: Influence score of stakeholder
 *         assessment_date:
 *           type: string
 *           format: date
 *           description: Assessment date of stakeholder
 *         priority_level:
 *           type: string
 *           description: Priority level of stakeholder
 *         assessment_notes:
 *           type: string
 *           description: Assessment notes of stakeholder
 *         organization:
 *           type: integer
 *           description: Organization of stakeholder
 *         department:
 *           type: integer
 *           description: Department of stakeholder
 *         deleted:
 *           type: integer
 *           description: Soft delete flag
 *         created_at:
 *           type: string
 *           format: date-time
 *           description: Creation timestamp
 *         created_by:
 *           type: integer
 *           description: Creator of the stakeholder identification
 *         updated_at:
 *           type: string
 *           format: date-time
 *           description: Last update timestamp
 *         updated_by:
 *           type: integer
 *           description: Last updater of the stakeholder identification
 *       example:
 *         id: 1
 *         stakeholder_type: 1
 *         stakeholder_category: "Internal"
 *         stakeholder_department_internal: 10
 *         stakeholder_name_internal: 101
 *         stakeholder_name_external: "John Doe"
 *         role_external: "Manager"
 *         email: "john.doe@example.com"
 *         phone: "1234567890"
 *         address: "123 Main St"
 *         classification: "High"
 *         engagement_level: "Medium"
 *         status: "Active"
 *         relationship_type: "Partner"
 *         interaction_frequency: "Weekly"
 *         influence_level: "High"
 *         connection_strength: "Strong"
 *         interest_score: "High"
 *         influence_score: "Medium"
 *         assessment_date: "2024-07-22"
 *         priority_level: "High"
 *         assessment_notes: "Key stakeholder in the project"
 *         organization: 1
 *         department: 2
 *         deleted: 0
 *         created_at: "2024-07-22T14:00:00Z"
 *         created_by: 1
 *         updated_at: "2024-07-22T14:00:00Z"
 *         updated_by: 1
 */

/**
 * @swagger
 * api/stakeholderIdentification/create-update-stakeholder-identification:
 *   post:
 *     summary: Create or update a stakeholder identification
 *     tags: [Stakeholder Identification]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             id: 1
 *             stakeholder_type: 1
 *             stakeholder_category: "Internal"
 *             stakeholder_department_internal: 10
 *             stakeholder_name_internal: 101
 *             stakeholder_name_external: "John Doe"
 *             role_external: "Manager"
 *             email: "john.doe@mail.com"
 *             phone: "1234567890"
 *             address: "123 Main St"
 *             classification: "High"
 *             engagement_level: "Medium"
 *             status: "Active"
 *             relationship_type: "Partner"
 *             interaction_frequency: "Weekly"
 *             influence_level: "High"
 *             connection_strength: "Strong"
 *             interest_score: "High"
 *             influence_score: "Medium"
 *             assessment_date: "2024-07-22"
 *             priority_level: "High"
 *             assessment_notes: "Key stakeholder in the project"
 *             organization: 1
 *             department: 2
 *             deleted: 0
 *             created_at: "2024-07-22T14:00:00Z"
 *             created_by: 1
 *             updated_at: "2024-07-22T14:00:00Z"
 *             updated_by: 1
 *     responses:
 *       200:
 *         description: The created/updated stakeholder identification
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/StakeHolderIdentification'
 */
stakeholderIdentificationRoutes.post(
  "/create-update-stakeholder-identification",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 225, "stakeholder_identification")(req, res, next);
  },
  controllerWrapper(createUpdateStakeholderIdentification)
);

/**
 * @swagger
 * api/stakeholderIdentification/get-stakeholder-identification:
 *   get:
 *     summary: Get all stakeholder identifications
 *     tags: [Stakeholder Identification]
 *     responses:
 *       200:
 *         description: List of all stakeholder identifications
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/StakeHolderIdentification'
 */
stakeholderIdentificationRoutes.get(
  "/get-stakeholder-identification",
  verifyToken,
  sideBarPermissionCheck("view", 225, "stakeholder_identification"),
  controllerWrapper(getStakeholderIdentification)
);

/**
 * @swagger
 * api/stakeholderIdentification/get-stakeholder-identification/{id}:
 *   get:
 *     summary: Get a stakeholder identification by id
 *     tags: [Stakeholder Identification]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The stakeholder identification id
 *     responses:
 *       200:
 *         description: The stakeholder identification by id
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/StakeHolderIdentification'
 */
stakeholderIdentificationRoutes.get(
  "/get-stakeholder-identification/:id",
  verifyToken,
  sideBarPermissionCheck("view", "225", "stakeholder_identification"),
  controllerWrapper(getStakeholderIdentification)
);

/**
 * @swagger
 * api/stakeholderIdentification/delete-stakeholder-identification/{id}:
 *   delete:
 *     summary: Delete a stakeholder identification by id
 *     tags: [Stakeholder Identification]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The stakeholder identification id
 *     responses:
 *       200:
 *         description: Stakeholder identification deleted
 */
stakeholderIdentificationRoutes.delete(
  "/delete-stakeholder-identification/:id",
  verifyToken,
  sideBarPermissionCheck("delete", "225", "stakeholder_identification"),
  controllerWrapper(deleteStakeholderIdentification)
);

/**
 * @swagger
 * /api/SOP/get-internal-stakeholder:
 *   get:
 *     summary: Get all internal stakeholder for dropdown
 *     tags: [Stakeholder Identification]
 */
stakeholderIdentificationRoutes.get(
  "/get-internal-stakeholder",
  verifyToken,
  controllerWrapper(getInternalStakeHolders)
);

export default stakeholderIdentificationRoutes;
