import { Router } from "express";
import {
  createUpdateTaskTemplate,
  getTaskTemplateList,
  deleteTaskTemplateList,
  approveRejectTemplateList,
} from "../controller/taskTemplateListController.js";

import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

/**
 * @swagger
 * tags:
 *   name: Task Template List
 *   description: API operations for managing task template list
 */

const taskTemplateListRoutes = Router();

/**
 * @swagger
 * /api/taskTemplateList/create-update-task-template:
 *   post:
 *     tags: [Task Template List]
 *     summary: Create or update an task template
 *     description: Creates a new task template or updates an existing one.
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: integer
 *                 description: The ID of the task template (for updates).
 *               name:
 *                 type: string
 *                 description: The name of the task template.
 *               visibility:
 *                 type: string
 *                 description: The visibility level of the task template.
 *               category:
 *                 type: string
 *                 description: The category of the task template.
 *               description:
 *                 type: string
 *                 description: A detailed description of the task template.
 *               subcategory:
 *                 type: string
 *                 description: The subcategory of the task template.
 *               attachment:
 *                 type: string
 *                 description: Any attachments related to the task template.
 *               assigned_to:
 *                 type: integer
 *                 description: The ID of the user assigned to the task template.
 *               dependencies:
 *                 type: string
 *                 description: Dependencies for the task template.
 *               task_type:
 *                 type: string
 *                 description: The type of the task related to the task template.
 *               module_name:
 *                 type: string
 *                 description: The module name related to the task template.
 *               sub_module_name:
 *                 type: string
 *                 description: The sub-module name related to the task template.
 *               record_name:
 *                 type: string
 *                 description: The record name related to the task template.
 *               task_title:
 *                 type: string
 *                 description: The title of the task related to the task template.
 *               task_description:
 *                 type: string
 *                 description: A description of the task related to the task template.
 *               priority:
 *                 type: string
 *                 description: The priority level of the task template.
 *               task_duration:
 *                 type: string
 *                 description: The duration of the task related to the task template.
 *               organization:
 *                 type: integer
 *                 description: The ID of the organization related to the task template.
 *               department:
 *                 type: integer
 *                 description: The ID of the department related to the task template.
 *             required:
 *               - name
 *               - category
 *           example:
 *             id: 1
 *             name: "Template Name"
 *             visibility: "public"
 *             category: "Category Name"
 *             description: "Detailed description of the task template."
 *             subcategory: "Subcategory Name"
 *             attachment: "attachment_file.pdf"
 *             assigned_to: 2
 *             dependencies: "None"
 *             task_type: "type1"
 *             module_name: "Module Name"
 *             sub_module_name: "Sub Module Name"
 *             record_name: "Record Name"
 *             task_title: "Task Title"
 *             task_description: "Task Description"
 *             priority: "High"
 *             task_duration: "2h"
 *             organization: 3
 *             department: 4
 *
 *     responses:
 *       200:
 *         description: Task template created or updated successfully.
 *       400:
 *         description: Bad request.
 *       401:
 *         description: Unauthorized.
 */

taskTemplateListRoutes.post(
  "/create-update-task-template",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      331,
      "task_template_list"
    )(req, res, next);
  },
  controllerWrapper(createUpdateTaskTemplate)
);

/**
 * @swagger
 * /api/taskTemplateList/get-task-template:
 *   get:
 *     tags: [Task Template List]
 *     summary: Get a list of task templates
 *     description: Retrieves a list of all task templates.
 *     responses:
 *       200:
 *         description: A list of task templates.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 type: object
 *                 properties:
 *                   id:
 *                     type: integer
 *                   name:
 *                     type: string
 *                   visibility:
 *                     type: string
 *                   category:
 *                     type: string
 *                   description:
 *                     type: string
 *                   subcategory:
 *                     type: string
 *                   attachment:
 *                     type: string
 *                   assigned_to:
 *                     type: integer
 *                   dependencies:
 *                     type: string
 *                   task_type:
 *                     type: string
 *                   module_name:
 *                     type: string
 *                   sub_module_name:
 *                     type: string
 *                   record_name:
 *                     type: string
 *                   task_title:
 *                     type: string
 *                   task_description:
 *                     type: string
 *                   priority:
 *                     type: string
 *                   task_duration:
 *                     type: string
 *                   organization:
 *                     type: integer
 *                   department:
 *                     type: integer
 *             example:
 *               - id: 1
 *                 name: "Template Name 1"
 *                 visibility: "public"
 *                 category: "Category 1"
 *                 description: "Description for template 1."
 *                 subcategory: "Subcategory 1"
 *                 attachment: "attachment1.pdf"
 *                 assigned_to: 2
 *                 dependencies: "None"
 *                 task_type: "type1"
 *                 module_name: "Module 1"
 *                 sub_module_name: "Sub Module 1"
 *                 record_name: "Record 1"
 *                 task_title: "Task Title 1"
 *                 task_description: "Task Description 1"
 *                 priority: "High"
 *                 task_duration: "1h"
 *                 organization: 3
 *                 department: 4
 *               - id: 2
 *                 name: "Template Name 2"
 *                 visibility: "private"
 *                 category: "Category 2"
 *                 description: "Description for template 2."
 *                 subcategory: "Subcategory 2"
 *                 attachment: "attachment2.pdf"
 *                 assigned_to: 3
 *                 dependencies: "Dependency 1"
 *                 task_type: "type2"
 *                 module_name: "Module 2"
 *                 sub_module_name: "Sub Module 2"
 *                 record_name: "Record 2"
 *                 task_title: "Task Title 2"
 *                 task_description: "Task Description 2"
 *                 priority: "Medium"
 *                 task_duration: "3h"
 *                 organization: 4
 *                 department: 5
 *       401:
 *         description: Unauthorized.
 */

taskTemplateListRoutes.get(
  "/get-task-template",
  verifyToken,
  sideBarPermissionCheck("view", 331, "task_template_list"),
  controllerWrapper(getTaskTemplateList)
);

/**
 * @swagger
 * /api/taskTemplateList/get-task-template/{id}:
 *   get:
 *     tags: [Task Template List]
 *     summary: Get an task template by ID
 *     description: Retrieves an task template by its ID.
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: The ID of the task template to retrieve.
 *         schema:
 *           type: integer
 *     responses:
 *       200:
 *         description: The task template details.
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 id:
 *                   type: integer
 *                 name:
 *                   type: string
 *                 visibility:
 *                   type: string
 *                 category:
 *                   type: string
 *                 description:
 *                   type: string
 *                 subcategory:
 *                   type: string
 *                 attachment:
 *                   type: string
 *                 assigned_to:
 *                   type: integer
 *                 dependencies:
 *                   type: string
 *                 task_type:
 *                   type: string
 *                 module_name:
 *                   type: string
 *                 sub_module_name:
 *                   type: string
 *                 record_name:
 *                   type: string
 *                 task_title:
 *                   type: string
 *                 task_description:
 *                   type: string
 *                 priority:
 *                   type: string
 *                 task_duration:
 *                   type: string
 *                 organization:
 *                   type: integer
 *                 department:
 *                   type: integer
 *             example:
 *               id: 1
 *               name: "Template Name"
 *               visibility: "public"
 *               category: "Category Name"
 *               description: "Detailed description of the task template."
 *               subcategory: "Subcategory Name"
 *               attachment: "attachment_file.pdf"
 *               assigned_to: 2
 *               dependencies: "None"
 *               task_type: "type1"
 *               module_name: "Module Name"
 *               sub_module_name: "Sub Module Name"
 *               record_name: "Record Name"
 *               task_title: "Task Title"
 *               task_description: "Task Description"
 *               priority: "High"
 *               task_duration: "2h"
 *               organization: 3
 *               department: 4
 *       401:
 *         description: Unauthorized.
 *       404:
 *         description: Task template not found.
 */

taskTemplateListRoutes.get(
  "/get-task-template/:id",
  verifyToken,
  sideBarPermissionCheck("view", 331, "task_template_list"),
  controllerWrapper(getTaskTemplateList)
);

/**
 * @swagger
 * /api/taskTemplateList/delete-task-template/{id}:
 *   delete:
 *     tags: [Task Template List]
 *     summary: Delete an task template by ID
 *     description: Deletes an task template by its ID.
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: The ID of the task template to delete.
 *         schema:
 *           type: integer
 *     responses:
 *       200:
 *         description: Task template deleted successfully.
 *       401:
 *         description: Unauthorized.
 *       404:
 *         description: Task template not found.
 */

taskTemplateListRoutes.delete(
  "/delete-task-template/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 331, "task_template_list"),
  controllerWrapper(deleteTaskTemplateList)
);

/**
 * @swagger
 * /api/taskTemplateList/approve-reject-task-template:
 *   post:
 *     tags: [Task Template List]
 *     summary: Approve or reject a action Template
 *     description: Approves or rejects a task template.
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               status:
 *                 type: string
 *                 description: The status of the task template (for updates) Approved , Rejected.
 *           example:
 *             status: "status"
 *     responses:
 *       200:
 *         description: Task template created or updated successfully.
 *       400:
 *         description: Bad request.
 *       401:
 *         description: Unauthorized.
 */

taskTemplateListRoutes.post(
  "/approve-reject-task-template",
  verifyToken,
  sideBarPermissionCheck("edit", 331, "task_template_list"),
  controllerWrapper(approveRejectTemplateList)
);

export default taskTemplateListRoutes;
