import db from "../db-config.js";

export async function createTriggersForAllTables() {
  try {
    // Fetch all tables and their primary keys
    const [tables] = await db.query(`
      SELECT kcu.TABLE_NAME, kcu.COLUMN_NAME AS primary_key
      FROM INFORMATION_SCHEMA.TABLES t
      JOIN INFORMATION_SCHEMA.KEY_COLUMN_USAGE kcu
          ON t.TABLE_NAME = kcu.TABLE_NAME
          AND t.TABLE_SCHEMA = DATABASE()
      WHERE t.TABLE_TYPE = 'BASE TABLE'
        AND kcu.CONSTRAINT_NAME = 'PRIMARY';
    `);

    for (const table of tables) {
      const { TABLE_NAME: tableName, primary_key: primaryKey } = table;
      if (tableName === "sops") {
        await db.query(`DROP TRIGGER IF EXISTS ${tableName}_AFTER_INSERT`);
        await db.query(`DROP TRIGGER IF EXISTS ${tableName}_AFTER_UPDATE`);

        console.log(`Dropped existing triggers for table: ${tableName}`);

        // Create AFTER INSERT trigger
        const insertTrigger = `
            CREATE TRIGGER ${tableName}_AFTER_INSERT
            AFTER INSERT ON ${tableName}
            FOR EACH ROW
            BEGIN
              INSERT INTO log_history (table_name, record_id, action, changed_fields)
              VALUES (
                '${tableName}',
                NEW.${primaryKey},
                'INSERT',
                NULL
              );
            END;
          `;
        await db.query(insertTrigger);
        console.log(`Created AFTER INSERT trigger for table: ${tableName}`);

        // Create AFTER UPDATE trigger
        const updateTrigger = `
            CREATE TRIGGER ${tableName}_AFTER_UPDATE
            AFTER UPDATE ON ${tableName}
            FOR EACH ROW
            BEGIN
              INSERT INTO log_history (table_name, record_id, action, changed_fields)
              VALUES (
                '${tableName}',
                OLD.${primaryKey},
                'UPDATE',
                JSON_OBJECT(${(await getChangedColumnsJSON(db, tableName)).join(
                  ", "
                )})
              );
            END;
          `;
        await db.query(updateTrigger);
        console.log(`Created AFTER UPDATE trigger for table: ${tableName}`);
      }
    }

    console.log("All triggers created successfully.");
  } catch (error) {
    console.error("Error creating triggers:", error.message);
  }
}

// Helper function to generate JSON_OBJECT keys for columns (for INSERT)
// async function getColumnJSON(db, tableName, prefix) {
//   const [columns] = await db.query(`
//     SELECT COLUMN_NAME
//     FROM INFORMATION_SCHEMA.COLUMNS
//     WHERE TABLE_NAME = '${tableName}' AND TABLE_SCHEMA = DATABASE();
//   `);

//   // Return key-value pairs in the format 'column_name', NEW.column_name
//   const columnJSON = columns.map((col) => `'${col.COLUMN_NAME}', ${prefix}.${col.COLUMN_NAME}`);

//   return columnJSON;
// }

// Helper function to generate JSON_OBJECT keys for changed columns (for UPDATE)
async function getChangedColumnsJSON(db, tableName) {
  const [columns] = await db.query(`
    SELECT COLUMN_NAME
    FROM INFORMATION_SCHEMA.COLUMNS
    WHERE TABLE_NAME = '${tableName}' AND TABLE_SCHEMA = DATABASE();
  `);

  const changedColumnsJSON = columns.map(
    (col) => `
      '${col.COLUMN_NAME}',
      IF(OLD.${col.COLUMN_NAME} != NEW.${col.COLUMN_NAME}, NEW.${col.COLUMN_NAME}, NULL)
    `
  );

  return changedColumnsJSON;
}

// createTriggersForAllTables();
