// MonitoringSchema.js
import sequelize from "./sequelize.js"; // Adjust the import path as necessary
import Organization from "./OrganizationSchema.js";
import User from "./UserSchema.js";
import Department from "./DepartmentSchema.js";
import EmployeeType from "./EmployeeTypeSchema.js"; // Assuming an EmployeeType schema exists
import ContractorRegistration from "./ContractorRegistrationSchema.js"; // Assuming a Contractor schema exists
import Location from "./LocationSchema.js"; // Assuming a Location schema exists
import { DataTypes } from "sequelize";
import Currency from "./CurrencySchema.js";

const Monitoring = sequelize.define(
  "Monitoring",
  {
    id: {
      type: DataTypes.INTEGER,
      autoIncrement: true,
      primaryKey: true,
    },
    monitoring_name: {
      type: DataTypes.STRING(250), // Monitoring Name
      allowNull: true,
    },
    monitoring_type: {
      type: DataTypes.INTEGER, // Dropdown for Monitoring Type
      allowNull: true,
    },
    planned_monitoring_start_date: {
      type: DataTypes.DATE, // Planned Monitoring Start Date
      allowNull: true,
    },
    planned_monitoring_end_date: {
      type: DataTypes.DATE, // Planned Monitoring End Date
      allowNull: true,
    },
    monitoring_source: {
      type: DataTypes.ENUM("Internal", "External"), // Source can be Internal or External
      allowNull: true,
    },
    // Internal Monitor
    internal_monitor_id: {
      type: DataTypes.INTEGER, // Monitor Name fetched from Employee List
      allowNull: true,
      references: {
        model: EmployeeType,
        key: "id",
      },
    },
    // External Monitor
    external_monitor_ids: {
      type: DataTypes.ARRAY(DataTypes.INTEGER), // Multi-select for external monitors (fetched from contractors)
      allowNull: true,
      references: {
        model: ContractorRegistration,
        key: "id",
      },
    },
    monitoring_owner: {
      type: DataTypes.STRING(250), // Monitoring Owner
      allowNull: true,
    },
    // Stakeholders Involved
    stakeholders_involved: {
      type: DataTypes.ENUM("Internal", "External"),
      allowNull: true,
    },
    internal_stakeholder_ids: {
      type: DataTypes.ARRAY(DataTypes.INTEGER), // Multi-select for internal stakeholders
      allowNull: true,
      references: {
        model: EmployeeType,
        key: "id",
      },
    },
    external_stakeholder_name: {
      type: DataTypes.STRING(250), // Free Text for External Stakeholder Name
      allowNull: true,
    },
    external_stakeholder_email: {
      type: DataTypes.STRING(250), // Email Input for External Stakeholder
      allowNull: true,
      validate: {
        isEmail: true, // Validate email format
      },
    },
    monitoring_location_ids: {
      type: DataTypes.ARRAY(DataTypes.INTEGER), // Multi-select Dropdown for Monitoring Locations
      allowNull: true,
      references: {
        model: Location,
        key: "id",
      },
    },
    monitoring_scope: {
      type: DataTypes.TEXT("long"), // Description Field for Monitoring Scope
      allowNull: true,
    },
    critical_business_processes: {
      type: DataTypes.TEXT, // Critical Business Processes
      allowNull: true,
    },

    // Technological Resources
    technological_resources: {
      type: DataTypes.TEXT, // Add More field for Technological Resources
      allowNull: true,
    },

    // Financial Resources
    financial_resources: {
      type: DataTypes.ENUM("Financial Resource Type", "Value", "Currency"),
      allowNull: true,
    },
    financial_resource_type: {
      type: DataTypes.ENUM("Operating Budget", "Capital Budget"), // Enum for Financial Resource Type
      allowNull: true,
    },
    value: {
      type: DataTypes.FLOAT, // Numerical Input for Financial Value
      allowNull: true,
      validate: {
        isFloat: true, // Validate float input
        min: 0, // Minimum value constraint
      },
    },
    currency_id: {
      type: DataTypes.INTEGER, // Dropdown for Currency, referencing prefilled dataset
      allowNull: true,
      references: {
        model: Currency,
        key: "id",
      },
    },
    evaluation_tool: {
      type: DataTypes.ENUM("Applicable", "Not Applicable"), // Dropdown for Evaluation Tool
      allowNull: true,
    },
    organization: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: Organization,
        key: "id",
      },
    },
    created_at: {
      type: DataTypes.DATE,
      allowNull: true,
      defaultValue: sequelize.literal("CURRENT_TIMESTAMP"),
    },
    created_by: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: User,
        key: "id",
      },
    },
    deleted: {
      type: DataTypes.TINYINT,
      allowNull: true,
      defaultValue: 0,
    },
    updated_at: {
      type: DataTypes.DATE,
      allowNull: true,
      defaultValue: sequelize.literal("CURRENT_TIMESTAMP"),
      onUpdate: sequelize.literal("CURRENT_TIMESTAMP"),
    },
    updated_by: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: User,
        key: "id",
      },
    },
    department: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: Department,
        key: "id",
      },
    },
  },
  {
    tableName: "monitoring",
    timestamps: false,
    charset: "utf8mb4",
    collate: "utf8mb4_general_ci",
  }
);

// Associations
Monitoring.belongsTo(EmployeeType, {
  as: "internalMonitor",
  foreignKey: "internal_monitor_id",
});
Monitoring.belongsToMany(ContractorRegistration, {
  as: "externalMonitors",
  through: "MonitoringExternalMonitors",
  foreignKey: "monitoring_id",
});
Monitoring.belongsToMany(EmployeeType, {
  as: "internalStakeholders",
  through: "MonitoringInternalStakeholders",
  foreignKey: "monitoring_id",
});
Monitoring.belongsToMany(Location, {
  as: "monitoringLocations",
  through: "MonitoringLocationMapping",
  foreignKey: "monitoring_id",
});
Monitoring.belongsTo(Currency, { as: "currency", foreignKey: "currency_id" });
Monitoring.belongsTo(Organization, { as: "org", foreignKey: "organization" });
Monitoring.belongsTo(User, { as: "creator", foreignKey: "created_by" });
Monitoring.belongsTo(User, { as: "updater", foreignKey: "updated_by" });
Monitoring.belongsTo(Department, { as: "dep", foreignKey: "department" });

export default Monitoring;
