import fs from "fs";
import path from "path";
import sequelize from "./sequelize.js";

// Function to load all Sequelize models
const loadModels = async () => {
  const schemaDir = "sequelize";
  const files = fs.readdirSync(schemaDir);
  const models = [];

  for (const file of files) {
    if (file.endsWith("Schema.js")) {
      const filePath = path
        .join(process.cwd(), schemaDir, file)
        .replace(/\\/g, "/");
      try {
        const module = await import(`file://${filePath}`);
        // console.log(`Loaded model from file: ${filePath}`); // Add this line for debugging
        models.push(module.default);
      } catch (error) {
        console.error(
          `Error loading model from ${file}:`,
          error.message,
          error.stack
        );
      }
    }
  }

  return models;
};
console.log(sequelize.models);
// Function to create tables without any constraints
// Function to create tables without any constraints in a synchronous manner
const createTablesWithoutConstraints = async (models) => {
  const transaction = await sequelize.transaction();
  try {
    for (const model of models) {
      const associations = { ...model.associations };
      model.associations = {};
      await sequelize.queryInterface.createTable(
        model.tableName,
        model.rawAttributes,
        { charset: "utf8mb4", collate: "utf8mb4_unicode_ci" },
        { transaction }
      );
      model.associations = associations;
    }
    await transaction.commit();
  } catch (error) {
    await transaction.rollback();
    console.error(`Error creating tables: ${error.message}`, error.stack);
  }
};

// Function to add constraints
const addConstraints = async (models) => {
  for (const model of models) {
    const attributes = model.rawAttributes;
    for (const attributeName in attributes) {
      const attribute = attributes[attributeName];
      if (attribute.references) {
        try {
          await sequelize.queryInterface.addConstraint(model.tableName, {
            fields: [attributeName],
            type: "foreign key",
            name: `fk_${model.tableName}_${attributeName}`,
            references: {
              table: attribute.references.model,
              field: attribute.references.key,
            },
            onDelete: attribute.onDelete || "CASCADE",
            onUpdate: attribute.onUpdate || "CASCADE",
          });
          console.log(
            `Foreign key constraint added for ${model.name}.${attributeName}`
          );
        } catch (error) {
          console.error(
            `Error adding foreign key constraint for ${model.name}.${attributeName}:`,
            error
          );
        }
      }
    }
  }
};

// Function to alter the table if columns `created_at` and `updated_at` exist
const alterTableIfColumnExists = async (tableName) => {
  // ... (keep this function as it is in your original code)
};



// Main execution block
(async () => {
  try {
    await sequelize.authenticate();
    console.log("Connection has been established successfully.");

    const models = await loadModels();
    console.log(`Loaded ${models.length} models.`);

    // Create tables without any constraints
    // await createTablesWithoutConstraints(models);

    // Add constraints
    // await addConstraints(models);

    // Fetch all table names from the database
    const tables = await sequelize.query(
      `
      SELECT TABLE_NAME
      FROM information_schema.tables
      WHERE table_schema = '${sequelize.config.database}';
    `,
      { type: sequelize.QueryTypes.SELECT }
    );
    // Loop through each table and alter the table if the column exists
    for (const table of tables) {
      // console.log(`Processing table '${table.TABLE_NAME}'...`);
      try {
        await alterTableIfColumnExists(table.TABLE_NAME);
      } catch (error) {
        console.error(`Error processing table '${table.TABLE_NAME}':`, error);
        // await handleTableError(table.table_name, error); // Handle table errors and attempt to fix
      }
    }

 

    console.log("All tables were processed successfully.");
  } catch (error) {
    console.error("Unable to connect to the database or alter tables:", error);
  } finally {
    await sequelize.close();
    // process.exit();
  }
})();
